//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.3

import wesual.Controls 1.0

Rectangle {
    id : timePicker

    readonly property string timeFormat  : "hh:mm"
    readonly property string enteredTime : hours.text + ":" + minutes.text
    property string selectedTime : {
        var now    = new Date();
        var locale = Qt.locale();
        return now.toLocaleTimeString(locale, timeFormat);
    }

    signal timePicked(string time)

    implicitWidth  : timeArea.implicitWidth
    implicitHeight : 26

    border {
        width : 1
        color : UI.color(UI.PrimaryControlBorder)
    }
    color : UI.color(UI.PrimaryBase)

    states : [
        State {
            name : "disabled"
            when : !enabled
            PropertyChanges {
                target : timePicker.border
                color  : UI.color(UI.DisabledBorder)
            }
            PropertyChanges {
                target : minutes
                color  : UI.color(UI.DisabledText)
            }
            PropertyChanges {
                target : hours
                color  : UI.color(UI.DisabledText)
            }
            PropertyChanges {
                target : separator
                color  : UI.color(UI.DisabledText)
            }
        },
        State {
            name : "selected"
            when : hours.activeFocus || minutes.activeFocus
            PropertyChanges {
                target : timePicker.border
                color  : UI.color(UI.ControlBorderPress)
            }
        },
        State {
            name : "hovered"
            when : timePickerMouseArea.containsMouse
            PropertyChanges {
                target : timePicker.border
                color  : UI.color(UI.ControlBorderHover)
            }
        }
    ]


    MouseArea {
        id : timePickerMouseArea
        anchors.fill : parent
        hoverEnabled : true
        enabled : timePicker.enabled
    }

    Row {
        id : timeArea
        anchors.margins  : 2
        anchors.centerIn : parent

        Rectangle {
            id : hoursWrapper

            width  : hours.implicitWidth
            height : hours.implicitHeight
            color : UI.color(UI.PrimaryBase)
            anchors.verticalCenter : parent.verticalCenter

            MouseArea {
                anchors.fill : parent
                onWheel : {
                    if (wheel.angleDelta.y > 0)
                        hours.text = p_.incrementedTime(hours.text, 24);
                    else
                        hours.text = p_.decrementedTime(hours.text, 24);
                }
                onClicked   : hours.forceActiveFocus()
                cursorShape : Qt.IBeamCursor
                enabled : timePicker.enabled
            }
            Text {
                id : hours
                focus : true
                font {
                    family : UI.fontFamily(UI.PrimaryFont)
                    weight : UI.fontWeight(UI.PrimaryFont)
                    pixelSize : 14
                }
                text : selectedTime.split(":")[0]
                horizontalAlignment : TextInput.AlignRight
                activeFocusOnTab : true

                Keys.onUpPressed    : text = p_.incrementedTime(text, 24)
                Keys.onDownPressed  : text = p_.decrementedTime(text, 24)
                Keys.onLeftPressed  : {}
                Keys.onRightPressed : minutes.focus = true
                Keys.onBacktabPressed  : {
                    var next = nextItemInFocusChain(false);
                    if (next)
                        next.focus = true;
                }
                Keys.onTabPressed : minutes.focus = true
                Keys.onEnterPressed   : p_.selectTime(hours)
                Keys.onReturnPressed  : p_.selectTime(hours)
                Keys.onEscapePressed  : p_.undoHours()
                Keys.onPressed : {
                    if (event.key === Qt.Key_Colon)
                        minutes.focus = true;

                    if (isNaN(parseInt(event.text))) {
                        return;
                    }

                    var hs = hours.text;
                    hs += event.text

                    hs = hs.slice(-2);

                    if (parseInt(hs) > 23) {
                        hours.text = hs.slice(-1);
                    } else {
                        hours.text = hs;
                    }
                }
                onActiveFocusChanged : {
                    if (!activeFocus)
                        p_.selectTime();
                }
            }

            states : State {
                name : "selected"
                when : hours.activeFocus
                PropertyChanges {
                    target : hoursWrapper
                    color  : UI.color(UI.TextSelection)
                }
                PropertyChanges {
                    target : hours
                    color  : UI.color(UI.SelectedTextColor)
                }
            }
        }
        Text {
            id : separator
            text : ":"
            anchors.verticalCenter : parent.verticalCenter
        }
        Rectangle {
            id : minutesWrapper

            width  : minutes.implicitWidth
            height : minutes.implicitHeight
            color : UI.color(UI.PrimaryBase)
            anchors.verticalCenter : parent.verticalCenter

            MouseArea {
                anchors.fill : parent
                onWheel : {
                    if (wheel.angleDelta.y > 0)
                        minutes.text = p_.incrementedTime(minutes.text, 60);
                    else
                        minutes.text = p_.decrementedTime(minutes.text, 60);
                }
                onClicked   : minutes.forceActiveFocus()
                cursorShape : Qt.IBeamCursor
                enabled : timePicker.enabled
            }
            Text {
                id : minutes
                focus : true
                text  : selectedTime.split(":")[1]
                font {
                    family : UI.fontFamily(UI.PrimaryFont)
                    weight : UI.fontWeight(UI.PrimaryFont)
                    pixelSize : 14
                }
                horizontalAlignment : TextInput.AlignRight
                activeFocusOnTab : true
                Keys.onUpPressed    : text = p_.incrementedTime(text, 60)
                Keys.onDownPressed  : text = p_.decrementedTime(text, 60)
                Keys.onLeftPressed  : hours.focus = true
                Keys.onRightPressed : {}
                Keys.onTabPressed   : {
                    var next = nextItemInFocusChain(true);
                    if (next)
                        next.focus = true;
                }
                Keys.onBacktabPressed : hours.focus = true
                Keys.onEnterPressed   : p_.selectTime(minutes)
                Keys.onReturnPressed  : p_.selectTime(minutes)
                Keys.onEscapePressed  : p_.undoMins()
                Keys.onPressed : {
                    if (isNaN(parseInt(event.text))) {
                        return;
                    }

                    var mins = minutes.text;
                    mins += event.text

                    mins = mins.slice(-2);

                    if (parseInt(mins) > 59) {
                        minutes.text = mins.slice(-1);
                    } else {
                        minutes.text = mins;
                    }
                }
                onActiveFocusChanged : {
                    if (!activeFocus)
                        p_.selectTime();
                }
            }

            states : State {
                name : "selected"
                when : minutes.activeFocus
                PropertyChanges {
                    target : minutesWrapper
                    color  : UI.color(UI.TextSelection)
                }
                PropertyChanges {
                    target : minutes
                    color  : UI.color(UI.SelectedTextColor)
                }
            }
        }
    }
    QtObject {
        id : p_

        function incrementedTime(text, modulus) {
            return (parseInt(text) + 1) % modulus;
        }

        function decrementedTime(text, modulus) {
            if (parseInt(text) === 0)
                return (modulus - 1);
            else
                return parseInt(text) - 1;
        }

        function selectTime(item) {
            if (minutes.text.length < 2)
                minutes.text = "0" + minutes.text;

            if (hours.text.length < 2) {
                hours.text = "0" + hours.text;
            }

            timePicker.selectedTime = timePicker.enteredTime;
            timePicker.timePicked(timePicker.selectedTime);

            hours.text = Qt.binding(function() {
                return selectedTime.split(":")[0];
            });
            minutes.text = Qt.binding(function() {
                return selectedTime.split(":")[1];
            });


            if (item)
                item.focus = false;
        }

        function undoHours() {
            hours.text = timePicker.selectedTime.split(":")[0];
        }

        function undoMins() {
            minutes.text = timePicker.selectedTime.split(":")[1];
        }
    }
}
